<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

require_once '../config/database.php';

try {
    $pdo = getDBConnection();
    
    // Get the request method
    $method = $_SERVER['REQUEST_METHOD'];
    
    if ($method === 'OPTIONS') {
        http_response_code(200);
        exit();
    }
    
    if ($method === 'GET') {
        $action = $_GET['action'] ?? '';
        
        switch ($action) {
            case 'check_date':
                // Check if a specific date is blocked
                $date = $_GET['date'] ?? '';
                if (empty($date)) {
                    throw new Exception('Date parameter is required');
                }
                
                $stmt = $pdo->prepare("SELECT * FROM blocked_dates WHERE block_date = ?");
                $stmt->execute([$date]);
                $blocked = $stmt->fetch();
                
                echo json_encode([
                    'success' => true,
                    'date' => $date,
                    'is_blocked' => !empty($blocked),
                    'block_reason' => $blocked['reason'] ?? null,
                    'blocked_at' => $blocked['created_at'] ?? null
                ]);
                break;
                
            case 'check_consultation_types':
                // Check which consultation types are enabled
                $stmt = $pdo->query("SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('audio_call_enabled', 'video_call_enabled', 'office_meeting_enabled')");
                $consultation_settings = [];
                while ($row = $stmt->fetch()) {
                    $consultation_settings[$row['setting_key']] = $row['setting_value'];
                }
                
                echo json_encode([
                    'success' => true,
                    'consultation_types' => [
                        'audio_call' => ($consultation_settings['audio_call_enabled'] ?? 'on') === 'on',
                        'video_call' => ($consultation_settings['video_call_enabled'] ?? 'on') === 'on',
                        'office_meeting' => ($consultation_settings['office_meeting_enabled'] ?? 'on') === 'on'
                    ]
                ]);
                break;
                
            case 'check_working_days':
                // Check which days are enabled for booking
                $stmt = $pdo->query("SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('monday_booking', 'tuesday_booking', 'wednesday_booking', 'thursday_booking', 'friday_booking', 'saturday_booking', 'sunday_booking')");
                $working_days = [];
                while ($row = $stmt->fetch()) {
                    $day = str_replace('_booking', '', $row['setting_key']);
                    $working_days[$day] = $row['setting_value'] === 'on';
                }
                
                echo json_encode([
                    'success' => true,
                    'working_days' => $working_days
                ]);
                break;
                
            case 'check_all':
                // Check all booking availability settings
                $date = $_GET['date'] ?? '';
                
                // Check if date is blocked
                $is_blocked = false;
                $block_reason = null;
                if (!empty($date)) {
                    $stmt = $pdo->prepare("SELECT * FROM blocked_dates WHERE block_date = ?");
                    $stmt->execute([$date]);
                    $blocked = $stmt->fetch();
                    $is_blocked = !empty($blocked);
                    $block_reason = $blocked['reason'] ?? null;
                }
                
                // Check consultation types
                $stmt = $pdo->query("SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('audio_call_enabled', 'video_call_enabled', 'office_meeting_enabled')");
                $consultation_settings = [];
                while ($row = $stmt->fetch()) {
                    $consultation_settings[$row['setting_key']] = $row['setting_value'];
                }
                
                // Check working days
                $stmt = $pdo->query("SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('monday_booking', 'tuesday_booking', 'wednesday_booking', 'thursday_booking', 'friday_booking', 'saturday_booking', 'sunday_booking')");
                $working_days = [];
                while ($row = $stmt->fetch()) {
                    $day = str_replace('_booking', '', $row['setting_key']);
                    $working_days[$day] = $row['setting_value'] === 'on';
                }
                
                // Check if the requested date falls on a working day
                $date_obj = new DateTime($date);
                $day_name = strtolower($date_obj->format('l'));
                $is_working_day = $working_days[$day_name] ?? false;
                
                echo json_encode([
                    'success' => true,
                    'date' => $date,
                    'is_blocked' => $is_blocked,
                    'block_reason' => $block_reason,
                    'is_working_day' => $is_working_day,
                    'consultation_types' => [
                        'audio_call' => ($consultation_settings['audio_call_enabled'] ?? 'on') === 'on',
                        'video_call' => ($consultation_settings['video_call_enabled'] ?? 'on') === 'on',
                        'office_meeting' => ($consultation_settings['office_meeting_enabled'] ?? 'on') === 'on'
                    ],
                    'working_days' => $working_days,
                    'can_book' => !$is_blocked && $is_working_day
                ]);
                break;
                
            default:
                throw new Exception('Invalid action parameter. Use: check_date, check_consultation_types, check_working_days, or check_all');
        }
        
    } else {
        throw new Exception('Only GET method is allowed');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?> 