<?php
session_start();
require_once '../config/database.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: login.php');
    exit;
}

$pdo = getDBConnection();
$success_message = '';
$error_message = '';

// Get help settings from database
$help_settings = [];
$stmt = $pdo->query("SELECT setting_key, setting_value FROM settings WHERE setting_key IN ('help_mobile_number', 'help_whatsapp_message', 'support_hours', 'support_email', 'language_support')");
while ($row = $stmt->fetch()) {
    $help_settings[$row['setting_key']] = $row['setting_value'];
}

// Set default values if settings don't exist
$help_mobile = $help_settings['help_mobile_number'] ?? '+91 77373 57224';
$whatsapp_message = $help_settings['help_whatsapp_message'] ?? 'Hi, I need help with the Astro Admin Panel. Can you please assist me?';
$support_hours = $help_settings['support_hours'] ?? '24/7 Available';
$support_email = $help_settings['support_email'] ?? 'support@astro.com';
$language_support = $help_settings['language_support'] ?? 'English, Hindi';

// Handle mobile number update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_mobile'])) {
    $userId = (int)$_POST['user_id'];
    $newMobile = $_POST['new_mobile'];
    
    // Validate mobile number
    if (preg_match('/^[0-9]{10}$/', $newMobile)) {
        $stmt = $pdo->prepare("UPDATE users SET mobile = ? WHERE id = ?");
        if ($stmt->execute([$newMobile, $userId])) {
            $success_message = "Mobile number updated successfully for user ID: $userId";
        } else {
            $error_message = "Error updating mobile number";
        }
    } else {
        $error_message = "Invalid mobile number format. Please enter 10 digits.";
    }
}

// Get users for mobile number management
$users_query = "SELECT id, name, mobile, city FROM users ORDER BY name ASC";
$users = $pdo->query($users_query)->fetchAll();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Help Utility - Astro Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            color: white;
            min-height: 100vh;
        }
        .card {
            background: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.2);
            border-radius: 15px;
        }
        .btn-help {
            background: linear-gradient(135deg, #25d366 0%, #128c7e 100%);
            border: none;
            border-radius: 25px;
            padding: 12px 25px;
            font-weight: 600;
            transition: all 0.3s;
        }
        .btn-help:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(37, 211, 102, 0.3);
        }
        .floating-help {
            position: fixed;
            bottom: 30px;
            right: 30px;
            z-index: 1000;
        }
        .floating-help .btn {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background: linear-gradient(135deg, #25d366 0%, #128c7e 100%);
            border: none;
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
            animation: pulse 2s infinite;
        }
        @keyframes pulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.1); }
            100% { transform: scale(1); }
        }
        .help-section {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
            border-left: 4px solid #25d366;
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <!-- Header -->
        <div class="row mb-4">
            <div class="col-12">
                <div class="d-flex justify-content-between align-items-center">
                    <h1><i class="fas fa-tools me-2"></i>Admin Help Utility</h1>
                    <div>
                        <a href="appointments.php" class="btn btn-outline-light me-2">
                            <i class="fas fa-arrow-left me-2"></i>Back to Appointments
                        </a>
                        <button class="btn btn-help" onclick="openWhatsApp()">
                            <i class="fab fa-whatsapp fa-lg"></i>
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <?php if ($success_message) { ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="fas fa-check-circle me-2"></i><?php echo htmlspecialchars($success_message); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php } ?>
        
        <?php if ($error_message) { ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="fas fa-exclamation-triangle me-2"></i><?php echo htmlspecialchars($error_message); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php } ?>

        <!-- Help Sections -->
        <div class="row">
            <div class="col-md-6">
                <div class="help-section">
                    <h4><i class="fas fa-mobile-alt me-2"></i>Mobile Number Management</h4>
                    <p class="text-muted">Update user mobile numbers directly from admin panel</p>
                    
                    <form method="POST" class="mt-3">
                        <div class="mb-3">
                            <label class="form-label">Select User</label>
                            <select name="user_id" class="form-select" required>
                                <option value="">Choose a user...</option>
                                <?php foreach ($users as $user): ?>
                                    <option value="<?php echo $user['id']; ?>">
                                        <?php echo htmlspecialchars($user['name']); ?> - 
                                        <?php echo htmlspecialchars($user['mobile']); ?> 
                                        (<?php echo htmlspecialchars($user['city']); ?>)
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">New Mobile Number</label>
                            <input type="text" name="new_mobile" class="form-control" 
                                   placeholder="Enter 10 digit mobile number" 
                                   pattern="[0-9]{10}" maxlength="10" required>
                        </div>
                        <button type="submit" name="update_mobile" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>Update Mobile Number
                        </button>
                    </form>
                </div>
            </div>
            
            <div class="col-md-6">
                <div class="help-section">
                    <h4><i class="fas fa-question-circle me-2"></i>Quick Help</h4>
                    <p class="text-muted">Common admin tasks and support</p>
                    
                    <div class="list-group list-group-flush">
                        <a href="#" class="list-group-item list-group-item-action bg-transparent text-white border-0" 
                           onclick="openWhatsAppWithMessage('Hi, I need help with user management')">
                            <i class="fas fa-users me-2"></i>User Management Help
                        </a>
                        <a href="#" class="list-group-item list-group-item-action bg-transparent text-white border-0"
                           onclick="openWhatsAppWithMessage('Hi, I need help with appointment management')">
                            <i class="fas fa-calendar me-2"></i>Appointment Management Help
                        </a>
                        <a href="#" class="list-group-item list-group-item-action bg-transparent text-white border-0"
                           onclick="openWhatsAppWithMessage('Hi, I need help with system configuration')">
                            <i class="fas fa-cog me-2"></i>System Configuration Help
                        </a>
                        <a href="#" class="list-group-item list-group-item-action bg-transparent text-white border-0"
                           onclick="openWhatsAppWithMessage('Hi, I need technical support')">
                            <i class="fas fa-wrench me-2"></i>Technical Support
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <!-- WhatsApp Support Section -->
        <div class="row mt-4">
            <div class="col-12">
                <div class="help-section">
                    <h4><i class="fab fa-whatsapp me-2"></i>24/7 WhatsApp Support</h4>
                    <p class="text-muted">Get instant help for any admin panel issues</p>
                    
                    <div class="row">
                        <div class="col-md-4">
                            <div class="text-center p-3">
                                <i class="fas fa-clock fa-2x text-warning mb-2"></i>
                                <h6>Quick Response</h6>
                                <small><?php echo htmlspecialchars($support_hours); ?></small>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="text-center p-3">
                                <i class="fas fa-language fa-2x text-info mb-2"></i>
                                <h6>Language Support</h6>
                                <small><?php echo htmlspecialchars($language_support); ?></small>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="text-center p-3">
                                <i class="fas fa-headset fa-2x text-success mb-2"></i>
                                <h6>Contact</h6>
                                <small><?php echo htmlspecialchars($help_mobile); ?></small>
                            </div>
                        </div>
                    </div>
                    
                    <div class="text-center mt-3">
                        <button class="btn btn-help btn-lg" onclick="openWhatsApp()">
                            <i class="fab fa-whatsapp me-2"></i>Start WhatsApp Chat
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Floating Help Button -->
    <div class="floating-help">
        <button class="btn" onclick="openWhatsApp()" title="WhatsApp Support">
            <i class="fab fa-whatsapp fa-lg text-white"></i>
        </button>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function openWhatsApp() {
            const message = '<?php echo addslashes($whatsapp_message); ?>';
            const whatsappUrl = `https://wa.me/<?php echo preg_replace('/[^0-9]/', '', $help_mobile); ?>?text=${encodeURIComponent(message)}`;
            window.open(whatsappUrl, '_blank');
        }
        
        function openWhatsAppWithMessage(message) {
            const whatsappUrl = `https://wa.me/<?php echo preg_replace('/[^0-9]/', '', $help_mobile); ?>?text=${encodeURIComponent(message)}`;
            window.open(whatsappUrl, '_blank');
        }
        
        // Auto-refresh page every 30 seconds to show latest data
        setInterval(function() {
            location.reload();
        }, 30000);
    </script>
</body>
</html> 